//	VorbisfileLoader.h - Sound file loaders for .ogg files, using vorbisfile.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef VORBISFILELOADER_H_
#define VORBISFILELOADER_H_

#include "SoundFileLoader.h"

#include "vorbis/vorbisfile.h"

///	Sound file loaders for .ogg files, using vorbisfile.
class VorbisfileLoader : public SoundFileLoader
{
  public:
	///	Constructor.
	VorbisfileLoader(const FilePath& path);
	///	Destructor.
	~VorbisfileLoader();

	///	Loads a chunk of the file into the passed-in buffer.
	int loadChunk(float *buffer,
				  int64_t& position,
				  int bufferSize,
				  int bufferOffset = 0);

	///	Returns the length of the sound file in samples.
	int64_t getFileLength() const {return fileLength;};
	///	Returns the sound file's samplerate.
	float getSamplerate() const {return fileSamplerate;};
	///	Returns the number of channels in the file.
	int getNumChannels() const {return fileNumChannels;};

	///	Returns the name of the sound file.
	const std::wstring getName() const {return filePath.getName();};

	///	Used to create VorbisfileLoaders
	class Creator : public SoundFileLoader::Creator
	{
	  public:
		///	Constructor.
		Creator() {};
		///	Destructor.
		~Creator() {};

		///	Returns a new instance of a VorbisfileLoader.
		SoundFileLoaderPtr createLoader(const FilePath& path) const
		{
			return SoundFileLoaderPtr(new VorbisfileLoader(path));
		};
	};

	///	The sole instance of Creator for this class.
	static Creator creator;
  private:
	///	The sound file's path.
	FilePath filePath;
	///	Handle to the sound file.
	OggVorbis_File vorbisFile;

	///	The length of the sound file.
	int64_t fileLength;
	///	The samplerate of the sound file.
	float fileSamplerate;
	///	The number of channels in the file.
	int fileNumChannels;

	///	True if vorbisFile was setup correctly in the constructor.
	bool valid;
};

#endif
