//	Sounder.h - Base class for game-specific Sounders.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef SOUNDER_H_
#define SOUNDER_H_

#include "c0xHeaders.h"

class SoundFileLoader;

typedef shared_ptr<SoundFileLoader> SoundFileLoaderPtr;

///	Base class for game-specific Sounders.
/*!
	This provides an interface for any audio operations your game may want to
	carry out.

	The idea is that a single instance of your subclass will exist throughout
	the game's lifespan, and you can implement any audio generation, playback
	etc. within it.
 */
class Sounder
{
  public:
	///	Constructor.
	Sounder():
	numInputChannels(0),
	numOutputChannels(2),
	samplerate(44100.0)
	{
		
	};
	///	Destructor.
	virtual ~Sounder() {};

	///	Called for every audio block.
	/*!
		This is where you do any audio generation/playback/calculations.
	 */
	virtual void getAudio(float *input, float *output, int numSamples) = 0;

	///	Called for each sound file resource by LoadingScreen.
	/*!
		This method should be implemented to handle sound file loading. It
		will be called from LoadingScreen for every sound file registered with
		the ResourceManager.
	 */
	virtual void loadSoundFileResource(SoundFileLoaderPtr soundFileLoader) = 0;

	///	Sets the number of channels you have to work with.
	void setNumChannels(int numInput, int numOutput)
	{
		numInputChannels = numInput;
		numOutputChannels = numOutput;
	};
	///	Sets the samplerate you're running at.
	void setSamplerate(double rate)
	{
		samplerate = rate;
	};

	///	Returns the number of input channels you have to work with.
	int getNumInputChannels() const {return numInputChannels;};
	///	Returns the number of input channels you have to work with.
	int getNumOutputChannels() const {return numOutputChannels;};
	///	Returns the current samplerate.
	double getSamplerate() const {return samplerate;};
  private:
	///	The number of input channels you have to work with.
	int numInputChannels;
	///	The number of output channels you have to work with.
	int numOutputChannels;
	///	The current samplerate.
	double samplerate;
};

typedef shared_ptr<Sounder> SounderPtr;

#endif
