//	PortAudioDevice.h - A PortAudio subclass of AudioDevice.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef PORTAUDIODEVICE_H_
#define PORTAUDIODEVICE_H_

#include "AudioDevice.h"

#include "portaudio.h"

///	A PortAudio subclass of AudioDevice.
class PortAudioDevice : public AudioDevice
{
  public:
	///	Constructor.
	PortAudioDevice();
	///	Destructor.
	~PortAudioDevice();

	///	Sets the callback to call during audio processing.
	void setCallback(AudioDeviceCallback *val);

	///	Sets the number of desired inputs and outputs.
	void setNumChannels(int inputs, int outputs);
	///	Sets the desired samplerate.
	void setSamplerate(double rate);
	///	Sets the audio devices to use for input and output.
	/*!
		The names of individual devices can be found via
		getAvailableInputDevices() and getAvailableOutputDevices(). Passing
		L"Default" will choose the default input/output device.
	 */
	void setDevices(const std::wstring& inDevice,
					const std::wstring& outDevice);
	///	Sets the audio API to use.
	/*!
		\param val Valid strings are:
		(Windows)
		DirectSound
		ASIO
		MME
		(Linux)
		ALSA
		JACK
		OSS
		(OSX)
		CoreAudio
	 */
	void setAPI(const std::wstring& val);

	///	Starts audio processing.
	/*!
		\return True if successful, false if there's a problem.
	 */
	bool start();
	///	Stops audio processing.
	void stop();

	///	Returns the number of input channels.
	int getNumInputs() const {return numInputs;};
	///	Returns the number of output channels.
	int getNumOutputs() const {return numOutputs;};
	///	Returns the samplerate.
	double getSamplerate() const {return samplerate;};
	///	Returns the current API.
	const std::wstring getAPI() const {return apiName;};

	///	Returns the available audio APIs as a vector of strings.
	const std::vector<std::wstring>& getAvailableAPIs() const {return apiList;};
	///	Returns the available inputs for the selected API (selected via setAPI()).
	const std::vector<std::wstring>& getAvailableInputDevices() const {return inputList;};
	///	Returns the available outputs for the selected API (selected via setAPI()).
	const std::vector<std::wstring>& getAvailableOutputDevices() const {return outputList;};
	///	Returns the number of available channels for the input device (index from getAvailableInputDevices()).
	int getChannelsForInputDevice(int index) const {return inputChannels[index];};
	///	Returns the number of available channels for the output device (index from getAvailableOutputDevices()).
	int getChannelsForOutputDevice(int index) const {return outputChannels[index];};

	///	Our PortAudio callback.
	static int portAudioCallback(const void *input,
								 void *output,
								 unsigned long frameCount,
								 const PaStreamCallbackTimeInfo *timeInfo,
								 PaStreamCallbackFlags statusFlags,
								 void *userData);
  private:
	///	Helper method. Returns the PaDeviceIndex for the named device.
	PaDeviceIndex getDeviceIndexFromName(const std::wstring& device,
										 bool isOutput);

	///	Our PortAudio stream.
	PaStream *stream;
	///	Our callback.
	AudioDeviceCallback *callback;
	
	///	The desired input device.
	std::wstring inputDevice;
	///	The desired output device.
	std::wstring outputDevice;
	///	The number of input channels.
	int numInputs;
	///	The number of output channels.
	int numOutputs;
	///	The samplerate.
	double samplerate;
	///	The API we're using.
	PaHostApiTypeId api;
	///	The name of the current API.
	std::wstring apiName;

	///	Returned by getAvailableAPIs().
	std::vector<std::wstring> apiList;
	///	Returned by getAvailableInputs().
	std::vector<std::wstring> inputList;
	///	Returned by getAvailableOutputs().
	std::vector<std::wstring> outputList;
	///	Returns by getChannelsForInputDevice().
	std::vector<int> inputChannels;
	///	Returns by getChannelsForOutputDevice().
	std::vector<int> outputChannels;

	///	Whether we're currently running or not.
	bool isRunning;
	///	If PortAudio has been initialised or not.
	bool initialised;
};

#endif
