//	GenericFileSounder.h - A Sounder which plays StreamingSounds.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef GENERICFILESOUNDER_H_
#define GENERICFILESOUNDER_H_

#include "StreamingSound.h"
#include "Sounder.h"
#include "Thread.h"

#include <string>
#include <map>

///	A Sounder which plays StreamingSounds.
class GenericFileSounder : public Sounder,
						   public Thread
{
  public:
	///	Constructor.
	GenericFileSounder();
	///	Destructor.
	~GenericFileSounder();
	
	///	Called for every audio block.
	void getAudio(float *input, float *output, int numSamples);

	///	Called for each sound file resource by LoadingScreen.
	void loadSoundFileResource(SoundFileLoaderPtr soundFileLoader);

	///	Used to stream audio off the disk.
	bool run();

	///	Plays a StreamingSound in a loop.
	/*!
		\param sound The name of the sound to play.

		This will tell a StreamingSound to play in a loop until it is replaced
		by another playMusic sound.
	 */
	void playMusic(const std::wstring& sound);
	///	Stops the music playing.
	void stopMusic();
	///	Triggers a StreamingSound to play as a one-shot.
	/*!
		\param sound The name of the sound to play.
		\param level How loud the sound should be.
		\param pan How the sound is panned (0->1; left->right).

		This will tell a StreamingSound to play, and stop when it reaches the
		end of its file.
	 */
	void triggerSound(const std::wstring& sound, float level = 1.0f, float pan = 0.5f);
	///	Triggers a sound to loop.
	/*!
		\return The index of the sound. This can be used to stop the sound
		looping with stopLoop().
	 */
	int triggerLoop(const std::wstring& sound,
					float level = 1.0f,
					float pan = 0.5f,
					bool fade = false);
	///	Stops the indexed loop playing.
	void stopLoop(int index);
	///	Stops all sound files playing.
	void stopAll();

	///	Returns true if the music track's already playing.
	bool getMusicIsPlaying() const {return (music > -1);};
  private:
	///	Struct representing a pre-loaded buffer.
	struct PreLoadedBuffer
	{
		///	Pointer to the buffer data.
		float *buffer;
		///	The size of the buffer data.
		int size;
	};

	///	The number of available StreamingSounds.
	enum
	{
		NumSounds = 128
	};

	///	Map of sound file names and their associated SoundFileLoaders.
	/*!
		Used for longer files.
	 */
	std::map<std::wstring, SoundFileLoaderPtr> soundFiles;
	///	Map of sound file names and their pre-loaded buffers.
	/*!
		Used for short files, which can reasonably be read entirely into
		memory.
	 */
	std::map<std::wstring, PreLoadedBuffer> preLoadedBuffers;

	///	The available StreamingSounds.
	StreamingSound sounds[NumSounds];

	///	Index of the current music StreamingSound.
	int music;

	///	Used to kill the thread.
	bool timeToStop;
};

#endif
