//	AudioDevice.h - Base class for all AudioDevice types.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef AUDIODEVICE_H_
#define AUDIODEVICE_H_

#include "c0xHeaders.h"

#include <string>
#include <vector>

//------------------------------------------------------------------------------
///	Audio callback class.
/*!
	Subclass this and register it with an AudioDevice to render audio to the
	soundcard.
 */
class AudioDeviceCallback
{
  public:
	///	Constructor.
	AudioDeviceCallback() {};
	///	Destructor.
	virtual ~AudioDeviceCallback() {};

	///	Called to generate audio to send to the soundcard.
	/*!
		input and output are interleaved buffers, dependent on how you
		called AudioDevice::setNumChannels(); by default they're stereo,
		but obviously you can have a different number of channels if
		your soundcard supports it.
	 */
	virtual void getAudio(float *input, float *output, int numSamples) = 0;
};

//------------------------------------------------------------------------------
///	Base class for all AudioDevice types.
class AudioDevice
{
  public:
	///	Constructor.
	AudioDevice() {};
	///	Destructor.
	virtual ~AudioDevice() {};

	///	Sets the callback to call during audio processing.
	/*!
		It's the caller's responsibility to delete the callback when you're done
		with it, not that of AudioDevice or its subclasses.
	 */
	virtual void setCallback(AudioDeviceCallback *val) = 0;

	///	Sets the number of desired inputs and outputs.
	virtual void setNumChannels(int inputs, int outputs) = 0;
	///	Sets the desired samplerate.
	virtual void setSamplerate(double rate) = 0;
	///	Sets the audio API to use.
	/*!
		This can be implemented for subclasses which offer a choice of audio
		APIs.
	 */
	virtual void setAPI(const std::wstring& val) = 0;
	///	Sets the audio devices to use for input and output.
	/*!
		The names of individual devices can be found via
		getAvailableInputDevices() and getAvailableOutputDevices(). Passing
		L"Default" will choose the default input/output device.
	 */
	virtual void setDevices(const std::wstring& inDevice,
							const std::wstring& outDevice) = 0;

	///	Starts audio processing.
	/*!
		\return True if successful, false if there's a problem.
	 */
	virtual bool start() = 0;
	///	Stops audio processing.
	virtual void stop() = 0;

	///	Returns the number of input channels.
	virtual int getNumInputs() const = 0;
	///	Returns the number of output channels.
	virtual int getNumOutputs() const = 0;
	///	Returns the samplerate.
	virtual double getSamplerate() const = 0;
	///	Returns the current API.
	virtual const std::wstring getAPI() const = 0;

	///	Returns the available audio APIs as a vector of strings.
	virtual const std::vector<std::wstring>& getAvailableAPIs() const = 0;
	///	Returns the available inputs for the selected API (selected via setAPI()).
	virtual const std::vector<std::wstring>& getAvailableInputDevices() const = 0;
	///	Returns the available outputs for the selected API (selected via setAPI()).
	virtual const std::vector<std::wstring>& getAvailableOutputDevices() const = 0;
	///	Returns the number of available channels for the input device (index from getAvailableInputDevices()).
	virtual int getChannelsForInputDevice(int index) const = 0;
	///	Returns the number of available channels for the output device (index from getAvailableOutputDevices()).
	virtual int getChannelsForOutputDevice(int index) const = 0;
};

//------------------------------------------------------------------------------
typedef shared_ptr<AudioDevice> AudioDevicePtr;

#endif
