//	LuaDrawer.h - Wrapper for passing a Drawer to a lua script.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2013  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef LUADRAWER_H_
#define LUADRAWER_H_

#include "ThreeFloats.h"
#include "c0xHeaders.h"
#include "TwoFloats.h"
#include "UTF8File.h"

#include "luabind/luabind.hpp"

#include <string>
#include <map>

struct lua_State;
class TextLayout;
class Drawer;

///	Wrapper for passing a Drawer to a lua script.
/*!
	NOTE: Make sure you call setDrawer before you call any of the drawing
	methods; we don't do any error checking in them.

	See Drawer for explanations of the various drawing methods.
 */
class LuaDrawer
{
  public:
	///	Constructor.
	LuaDrawer();
	///	Destructor.
	~LuaDrawer();

	///	Sets the Drawer this instance wraps.
	void setDrawer(shared_ptr<Drawer>& d);
	///	Returns true if the Drawer has been set.
	bool getDrawer() const {return (drawer != 0);};
	///	Registers the class' methods with luabind (also TextLayout).
	static void registerLuaMethods(lua_State *luaState);


	//--Drawer methods---------------------------------------------------------
	///	Toggles whether to clear the screen between frames.
	void setClearBetweenFrames(bool val);

	///	Associates a shader parameter (uniform) with the current shader.
	void registerShaderParam(const std::string& name);

	///	Activates a shader to be used for the following operations.
	void useShader(const std::string& shader);
	///	Sets a shader parameter (uniform) for the current shader.
	void setShaderParam(const std::string& param, float val);

	///	Draws an image at the desired position.
	void drawImage(const std::string& name,
				   const TwoFloats& position,
				   const TwoFloats& size,
				   const ThreeFloats& colour,
				   float alpha);
	///	Draws a rotated image at the desired position.
	void drawRotImage(const std::string& name,
					  const TwoFloats& position,
					  float angle,
					  const TwoFloats& pivot,
					  const TwoFloats& size,
					  const ThreeFloats& colour,
					  float alpha);
	///	Draws part of an image at the desired position.
	void drawSubImage(const std::string& name,
					  const TwoFloats& position,
					  const TwoFloats& subPos,
					  const TwoFloats& subSize,
					  const TwoFloats& size,
					  const ThreeFloats& colour,
					  float alpha);
	///	Draws a single image tiled multiple times within the desired rect.
	void drawTiledImage(const std::string& name,
						const TwoFloats& position,
						const TwoFloats& size,
						const TwoFloats& tilePos,
						const TwoFloats& tileNum,
						const ThreeFloats& colour,
						float alpha);
	///	Draws a string of text with the selected font.
	void drawText(const std::string& text,
				  const std::string& font,
				  const TwoFloats& position,
				  float boxWidth,
				  int justification,
				  float size,
				  const ThreeFloats& colour,
				  float alpha);
	///	Draws a string of text with the selected font, caching the TextLayout.
	void drawTextCached(const std::string& text,
					    const std::string& font,
					    TextLayout& layout,
					    const TwoFloats& position,
					    float boxWidth,
					    int justification,
					    float size,
					    const ThreeFloats& colour,
					    float alpha);
	///	Draws a single line.
	void drawLine(const std::string& image,
				  const TwoFloats& start,
				  const TwoFloats& end,
				  const ThreeFloats& colour,
				  float alpha);
	///	Draws a connected line.
	void drawConnectedLine(const std::string& image,
						   const luabind::object& points,
						   const luabind::object& pointsScale,
						   const luabind::object& pointsColour,
						   const luabind::object& pointsAlpha);
	///	Fills a rectangle with the desired colour.
	void fillRect(const TwoFloats& position,
				  const TwoFloats& size,
				  const ThreeFloats& colour,
				  float alpha);
	///	Fills a rectangle with a horizontal gradient.
	void fillGradient(const TwoFloats& position,
					  const TwoFloats& size,
					  const ThreeFloats& colourTop,
					  const ThreeFloats& colourBottom,
					  float alphaTop,
					  float alphaBottom);

	///	Translates all subsequent drawing commands by the specified offset.
	void translate(const TwoFloats& offset);
	///	Scales all subsequent drawing commands by the specified amounts.
	void scale(const TwoFloats& size);
	///	Rotates all subsequent drawing commands by the specified angle.
	void rotate(float angle, const TwoFloats& pivot);

	///	Specifies a viewport to restrict drawing to.
	void setViewport(const TwoFloats& position, const TwoFloats& size);
	///	Resets the viewport.
	void resetViewport();

	///	Used to draw to a frame buffer instead of the screen.
	void startFrameBufferDrawing();
	///	Used to draw to a frame buffer instead of the screen.
	void endFrameBufferDrawing();
	///	Used to e.g. apply post-processing etc. to the contents of a frame buffer image.
	const std::string getFrameBufferImage() const;

	///	Returns the size of the drawing area.
	const TwoFloats getDrawingAreaSize() const;
	///	Returns the size of the window.
	const TwoFloats getWindowSize() const;
  private:
	///	Helper method. Checks if the passed-in name has already been converted, converts it if not.
	inline const std::wstring& getWideName(const std::string& name)
	{
		std::map<std::string, std::wstring>::iterator it;

		it = cachedNames.find(name);
		if(it == cachedNames.end())
		{
			std::wstring tempstr;

			UTF8File::charToWstring(name, tempstr);
			cachedNames.insert(std::make_pair(name, tempstr));
			it = cachedNames.find(name);
		}

		return it->second;
	};

	///	Our copy of the actual Drawer.
	shared_ptr<Drawer> drawer;

	///	We cache the narrow->wide char names we receive so we don't have to convert them every time.
	std::map<std::string, std::wstring> cachedNames;

	///	Temporary vectors used by drawConnectedLine().
	std::vector<TwoFloats> linePoints;
	///	Temporary vectors used by drawConnectedLine().
	std::vector<float> lineScales;
	///	Temporary vectors used by drawConnectedLine().
	std::vector<ThreeFloats> lineColours;
	///	Temporary vectors used by drawConnectedLine().
	std::vector<float> lineAlphas;
};

#endif
