//	ThreeFloats.h - Simple struct containing three floats.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//	----------------------------------------------------------------------------

#ifndef THREEFLOATS_H_
#define THREEFLOATS_H_

#include <cmath>

///	Simple struct containing three floats.
struct ThreeFloats
{
	float x;
	float y;
	float z;

	///	Default constructor.
	ThreeFloats()
	{
		x = y = z = 0.0f;
	}

	///	Constructor, all members set to val.
	ThreeFloats(float val)
	{
		x = y = z = val;
	}

	///	Constructor.
	ThreeFloats(float xVal, float yVal, float zVal)
	{
		x = xVal;
		y = yVal;
		z = zVal;
	}

	///	Assignment operator.
	ThreeFloats& operator=(const ThreeFloats& op2)
	{
		x = op2.x;
		y = op2.y;
		z = op2.z;
		return *this;
	}

	///	Assignment operator (assigns all members the value of val).
	ThreeFloats& operator=(float val)
	{
		x = val;
		y = val;
		z = val;
		return *this;
	}

	///	Addition operator.
	ThreeFloats operator+(const ThreeFloats& op2) const
	{
		ThreeFloats retval;

		retval.x = x + op2.x;
		retval.y = y + op2.y;
		retval.z = z + op2.z;

		return retval;
	}

	///	Subtraction operator.
	ThreeFloats operator-(const ThreeFloats& op2) const
	{
		ThreeFloats retval;

		retval.x = x - op2.x;
		retval.y = y - op2.y;
		retval.z = z - op2.z;

		return retval;
	}

	///	Multiplication operator.
	ThreeFloats operator*(const ThreeFloats& op2) const
	{
		ThreeFloats retval;

		retval.x = x * op2.x;
		retval.y = y * op2.y;
		retval.z = z * op2.z;

		return retval;
	}

	///	Multiplication operator (float).
	ThreeFloats operator*(float val) const
	{
		ThreeFloats retval;

		retval.x = x * val;
		retval.y = y * val;
		retval.z = z * val;

		return retval;
	}

	///	Division operator (ThreeFloats).
	ThreeFloats operator/(const ThreeFloats& op2) const
	{
		ThreeFloats retval;

		retval.x = x * op2.x;
		retval.y = y * op2.y;
		retval.z = z * op2.z;

		return retval;
	}

	///	Division operator (float).
	ThreeFloats operator/(float val) const
	{
		ThreeFloats retval;

		retval.x = x / val;
		retval.y = y / val;
		retval.z = z / val;

		return retval;
	}

	///	Addition assignment operator.
	ThreeFloats& operator+=(const ThreeFloats& op2)
	{
		x += op2.x;
		y += op2.y;
		z += op2.z;

		return *this;
	}

	///	Subtraction assignment operator.
	ThreeFloats& operator-=(const ThreeFloats& op2)
	{
		x -= op2.x;
		y -= op2.y;
		z -= op2.z;

		return *this;
	}

	///	Multiplication assignment operator.
	ThreeFloats& operator*=(const ThreeFloats& op2)
	{
		x *= op2.x;
		y *= op2.y;
		z *= op2.z;

		return *this;
	}

	///	Multiplication assignment operator.
	ThreeFloats& operator*=(float op2)
	{
		x *= op2;
		y *= op2;
		z *= op2;

		return *this;
	}

	///	Division assignment operator.
	ThreeFloats& operator/=(const ThreeFloats& op2)
	{
		x /= op2.x;
		y /= op2.y;
		z /= op2.z;

		return *this;
	}

	///	Division assignment operator.
	ThreeFloats& operator/=(float op2)
	{
		x /= op2;
		y /= op2;
		z /= op2;

		return *this;
	}

	///	Equality operator.
	bool operator==(const ThreeFloats& op2) const
	{
		return (x==op2.x)&&(y==op2.y)&&(z==op2.z);
	}

	///	Inequality operator.
	bool operator!=(const ThreeFloats& op2) const
	{
		return (x!=op2.x)||(y!=op2.y)||(z!=op2.z);
	}

	///	Returns the distance between two ThreeFloats.
	float getDistance(const ThreeFloats& op2) const
	{
		float tempX = op2.x - x;
		float tempY = op2.y - y;
		float tempZ = op2.z - z;

		return sqrtf((tempX*tempX) + (tempY*tempY) + (tempZ*tempZ));
	};

	friend ThreeFloats operator*(float op1, const ThreeFloats& op2);
};

#endif
