//	CriticalSection.h - Simple class representing a critical section.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef CRITICALSECTION_H_
#define CRITICALSECTION_H_

#include "c0xHeaders.h"

class BackendCritSec;
typedef shared_ptr<BackendCritSec> BackendCritSecPtr;

//------------------------------------------------------------------------------
///	Base class for any backend-specific CriticalSection implementations.
class BackendCritSec
{
  public:
	///	Constructor.
	BackendCritSec() {};
	///	Destructor.
	virtual ~BackendCritSec() {};

	///	(Waits until the critical section is free and) Locks the Critical Section.
	/*!
		\return True if it was successfully locked, false otherwise.
	 */
	virtual bool lock() = 0;
	///	Unlocks the critical section.
	virtual void unlock() = 0;

	///	Simple class used to construct an instance of a BackendCritSec subclass.
	class Creator
	{
	  public:
		///	Constructor.
		Creator() {};
		///	Destructor.
		virtual ~Creator() {};

		///	Returns a new instance of a BackendCritSec subclass.
		virtual BackendCritSecPtr createBackendCritSec() const = 0;
	};
};

//------------------------------------------------------------------------------
///	Simple class representing a critical section.
class CriticalSection
{
  public:
	///	Constructor.
	CriticalSection();
	///	Destructor.
	~CriticalSection();

	///	(Waits until the critical section is free and) Locks the Critical Section.
	/*!
		\return True if it was successfully locked, false otherwise.
	 */
	bool lock();
	///	Unlocks the critical section.
	void unlock();

	///	Used to set the BackendCritSec creator.
	/*!
		This *MUST* be called on startup, or CriticalSection will refuse to work.

		This method is necessary because different backends may have different
		thread implementations.

		This method may only be called *once*; multiple calls constitutes an
		error.
	 */
	static void setBackendCritSecCreator(BackendCritSec::Creator *creator);
  private:
	///	The backend-specific implementation of the CriticalSection.
	BackendCritSecPtr backendCritSec;

	///	Used to create the correct BackendCritSec for the current backend.
	static BackendCritSec::Creator *backendCritSecCreator;
};

//------------------------------------------------------------------------------
///	Simple scoped lock to be used with CriticalSection.
class ScopedLock
{
  public:
	///	Constructor.
	ScopedLock(CriticalSection& criticalSection):
	critSec(criticalSection)
	{
		critSec.lock();
	};
	///	Destructor.
	~ScopedLock()
	{
		critSec.unlock();
	};
  private:
	///	Our reference to the CriticalSection.
	CriticalSection& critSec;
};

#endif
