//	VerticalSlider.cpp - A vertical slider widget.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#include "drawing/Drawer.h"
#include "VerticalSlider.h"
#include "MouseCursor.h"

//------------------------------------------------------------------------------
VerticalSlider::VerticalSlider(const std::wstring& sliderImage):
Widget(),
ChangeBroadcaster(),
image(sliderImage),
value(0.0f),
handleSize(1.0f),
dragging(false),
mouseIsOver(false),
sizeIsSet(false)
{
	
}

//------------------------------------------------------------------------------
VerticalSlider::~VerticalSlider()
{
	
}

//------------------------------------------------------------------------------
void VerticalSlider::draw(DrawerPtr& d)
{
	TwoFloats tempPos(getPosition());

	if(!sizeIsSet)
	{
		const Drawer::Image * const tempIm = d->getInternalImageData(image);
		if(tempIm)
		{
			handleSize = tempIm->size.y;
			sizeIsSet = true;
		}
	}

	tempPos.y += (1.0f-value) * getSize().y;
	tempPos.y -= (handleSize*0.5f);

	d->drawImage(image, tempPos);
}

//------------------------------------------------------------------------------
void VerticalSlider::mouseMove(const TwoFloats& pos)
{
	float left, right;
	const float w = getSize().y;

	mouseIsOver = true;
	MouseCursor::getInstance().setImage(true);

	left = getPosition().y;
	right = left + w;

	if(dragging)
	{
		if((pos.y >= left) && (pos.y < right))
			value = 1.0f-((pos.y-left)/w);
		else if(pos.y < left)
			value = 1.0f;
		else if(pos.y > right)
			value = 0.0f;
		callChangeCallbacks(this, value);
	}
}

//------------------------------------------------------------------------------
void VerticalSlider::mouseButton(bool val, const TwoFloats& pos)
{
	float left, right;
	const float w = getSize().y;

	if(val)
	{
		left = getPosition().y;
		right = left + w;

		if((pos.y >= left) && (pos.y < right))
		{
			value = 1.0f-((pos.y-left)/w);
			dragging = true;
		}
	}
	else
	{
		dragging = false;
		callFinishedChanging(this);
	}
}

//------------------------------------------------------------------------------
void VerticalSlider::mouseLeave(const TwoFloats& pos)
{
	float left, right;
	const float w = getSize().y;

	if(!dragging && mouseIsOver)
	{
		mouseIsOver = false;
		MouseCursor::getInstance().setImage(false);
	}
	else if(dragging)
	{
		left = getPosition().y;
		right = left + w;

		if((pos.y >= left) && (pos.y < right))
			value = 1.0f-((pos.y-left)/w);
		else if(pos.y < left)
			value = 1.0f;
		else if(pos.y > right)
			value = 0.0f;
		callChangeCallbacks(this, value);
	}
}

//------------------------------------------------------------------------------
void VerticalSlider::setValue(float val)
{
	value = val;
}
