//	MouseCursor.cpp - Singleton class for drawing a mouse cursor.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#include "drawing/ImageLoader.h"
#include "drawing/Drawer.h"
#include "MouseCursor.h"

//------------------------------------------------------------------------------
MouseCursor& MouseCursor::getInstance()
{
	static MouseCursor retval;

	return retval;
}

//------------------------------------------------------------------------------
void MouseCursor::draw(DrawerPtr& d)
{
	TwoFloats tempPos(position);

	if(!defaultImageLoaded)
	{
		ImageData data;
		ImageLoader imLoad;

		data = imLoad.loadImage(MouseImages::mousecursor_png,
								MouseImages::mousecursor_pngSize);
		d->loadSingleImage(data.data, data.width, data.height, L"guiDefaultCursor");
		pointer = L"guiDefaultCursor";

		data = imLoad.loadImage(MouseImages::mousecursorhand_png,
								MouseImages::mousecursorhand_pngSize);
		d->loadSingleImage(data.data, data.width, data.height, L"guiDefaultCursorHand");
		hand = L"guiDefaultCursorHand";

		data = imLoad.loadImage(MouseImages::mousecursorhandclosed_png,
								MouseImages::mousecursorhandclosed_pngSize);
		d->loadSingleImage(data.data, data.width, data.height, L"guiDefaultCursorHandClosed");
		handClosed = L"guiDefaultCursorHandClosed";

		defaultImageLoaded = true;
	}

	if(image)
	{
		tempPos.x -= 14.0f;
		tempPos.y -= 8.0f;
		if(mouseIsDown)
			d->drawImage(handClosed, tempPos);
		else
			d->drawImage(hand, tempPos);
	}
	else
	{
		tempPos.x -= 7.0f;
		tempPos.y -= 8.0f;
		d->drawImage(pointer, tempPos);
	}
}

//------------------------------------------------------------------------------
void MouseCursor::setImage(bool val)
{
	image = val;
}

//------------------------------------------------------------------------------
void MouseCursor::setImages(const std::wstring& pointerImage,
						    const std::wstring& handImage,
						    const std::wstring& handClosedImage)
{
	pointer = pointerImage;
	hand = handImage;
	handClosed = handClosedImage;

	defaultImageLoaded = true;
}

//------------------------------------------------------------------------------
void MouseCursor::mouseMove(const TwoFloats& pos)
{
	position = pos;
}

//------------------------------------------------------------------------------
void MouseCursor::mouseButton(bool val, const TwoFloats& pos)
{
	mouseIsDown = val;
}

//------------------------------------------------------------------------------
MouseCursor::MouseCursor():
image(false),
defaultImageLoaded(false)
{
	
}

//------------------------------------------------------------------------------
MouseCursor::~MouseCursor()
{
	
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
static const unsigned char temp1[] = {137,80,78,71,13,10,26,10,0,0,0,13,73,72,68,82,0,0,0,64,0,0,0,64,8,6,0,0,0,170,105,113,222,0,0,0,4,115,66,73,
  84,8,8,8,8,124,8,100,136,0,0,0,9,112,72,89,115,0,0,13,215,0,0,13,215,1,66,40,155,120,0,0,0,25,116,69,88,116,83,111,
  102,116,119,97,114,101,0,119,119,119,46,105,110,107,115,99,97,112,101,46,111,114,103,155,238,60,26,0,0,10,223,73,68,65,84,120,156,229,155,93,
  108,84,199,21,199,127,187,6,127,128,193,164,196,105,93,242,33,220,186,234,183,85,154,22,53,54,173,148,154,54,113,65,164,126,0,149,186,145,106,33,
  17,43,15,164,34,106,8,79,173,154,135,170,70,137,224,161,138,234,23,8,173,170,84,68,9,34,169,171,146,16,67,19,130,43,139,64,33,101,205,135,
  195,174,77,109,130,155,218,222,101,215,56,187,59,125,56,51,247,142,47,251,121,215,216,180,57,210,200,203,204,189,115,206,249,207,153,115,206,156,185,4,
  148,82,124,156,105,65,174,193,64,32,16,240,51,169,250,31,66,53,224,149,85,43,93,6,252,16,88,15,124,26,168,211,173,6,184,14,76,2,49,32,
  10,92,3,46,0,33,171,141,122,25,221,174,160,56,0,104,197,203,129,199,128,159,1,247,149,48,239,4,240,46,240,55,224,24,240,14,16,183,31,184,
  109,0,209,114,4,116,123,2,80,183,160,77,35,32,252,26,248,22,16,180,120,6,148,82,204,87,51,66,128,172,254,0,165,173,124,161,52,8,252,9,
  248,35,112,86,131,4,204,189,101,24,0,130,192,38,224,15,246,96,107,107,43,27,55,110,100,197,138,21,212,213,213,81,83,83,67,44,22,99,114,114,
  146,104,52,74,52,26,101,100,100,132,129,129,1,66,161,16,3,3,3,132,195,97,138,212,225,172,230,251,59,96,28,13,198,92,2,17,4,42,144,213,
  112,204,182,181,181,85,249,161,120,60,174,250,250,250,84,87,87,151,90,183,110,157,90,182,108,89,161,219,228,63,192,47,129,229,136,19,158,147,237,1,
  18,10,23,35,206,202,17,104,239,222,189,190,0,240,82,42,149,82,39,79,158,84,207,62,251,172,90,179,102,141,10,4,2,249,128,24,7,126,5,220,
  57,23,64,128,236,253,101,192,121,91,144,195,135,15,207,10,0,94,138,68,34,170,171,171,75,173,90,181,42,31,16,35,64,139,94,160,224,173,2,2,
  160,18,184,11,9,93,142,0,103,207,158,189,37,0,216,20,10,133,84,103,103,167,170,168,168,200,6,66,82,111,139,74,13,196,172,131,0,176,8,88,
  1,164,108,230,67,67,67,183,28,0,67,195,195,195,106,219,182,109,170,170,170,42,27,16,175,3,247,216,214,48,155,0,84,35,161,239,186,205,244,220,
  185,115,115,6,128,161,209,209,81,213,222,222,158,13,132,65,160,65,111,217,178,217,2,1,96,9,80,143,164,180,14,195,190,190,190,57,7,192,80,119,
  119,183,170,172,172,204,4,194,37,224,115,72,212,154,21,16,140,115,9,2,9,44,138,70,163,204,23,109,217,178,133,19,39,78,208,208,208,224,29,170,
  7,94,67,156,246,2,32,232,247,192,102,200,78,73,111,27,0,0,26,27,27,233,239,239,167,185,185,217,59,244,89,96,39,80,133,118,140,165,128,16,
  180,126,79,216,3,35,35,35,126,231,156,53,90,186,116,41,61,61,61,52,53,53,121,135,30,3,86,34,91,161,36,16,12,0,10,24,182,7,6,6,
  6,252,204,55,235,84,93,93,205,238,221,187,189,221,229,136,239,90,12,44,100,230,66,22,69,65,92,7,19,177,7,66,161,144,223,57,103,149,18,137,
  4,143,63,254,184,183,59,137,11,130,227,16,253,88,129,1,32,13,132,237,129,219,193,2,18,137,4,155,54,109,162,175,175,207,59,52,0,220,129,7,
  0,63,60,108,0,6,237,129,112,56,76,34,145,200,248,210,92,80,36,18,161,185,185,153,67,135,14,121,135,254,13,252,5,81,126,17,86,94,224,135,
  79,16,81,62,133,228,1,142,198,74,41,206,156,57,227,103,206,146,233,232,209,163,220,127,255,253,156,60,121,210,59,148,192,61,180,45,68,20,247,189,
  255,177,94,78,1,31,1,239,219,131,199,142,29,43,101,110,95,116,228,200,17,30,122,232,33,174,93,187,230,29,186,1,244,33,25,107,82,183,20,178,
  128,190,41,168,39,49,0,188,103,15,30,61,122,180,148,185,139,166,193,193,65,54,108,216,192,212,212,148,119,104,2,120,27,24,67,138,177,147,72,65,
  54,129,148,219,204,57,166,104,242,90,192,12,155,127,235,173,183,72,167,75,2,184,40,234,238,238,38,22,139,121,187,175,0,39,144,130,201,36,226,3,
  198,128,15,17,16,74,6,192,248,128,105,224,148,254,13,192,248,248,56,167,79,159,246,51,175,47,122,241,197,23,189,93,35,192,63,144,85,159,0,62,
  208,125,35,8,8,147,192,20,26,0,165,84,209,32,152,40,96,44,32,10,92,182,31,232,237,237,45,118,78,223,212,216,216,232,237,186,19,73,118,140,
  242,87,144,132,109,20,177,128,56,34,183,111,51,181,1,72,34,142,230,156,253,192,203,47,191,236,119,238,162,169,189,189,221,219,181,16,88,3,124,5,
  23,128,89,91,125,67,65,220,170,208,215,128,167,177,142,160,129,64,64,69,34,145,57,57,6,167,211,105,213,209,209,145,173,30,48,4,252,20,248,12,
  82,56,117,14,67,165,28,135,65,18,136,10,224,19,192,23,128,117,26,97,135,121,87,87,215,156,0,160,148,82,201,100,50,87,81,36,5,252,94,131,
  96,103,128,190,65,48,0,44,0,150,34,231,237,239,2,61,54,227,85,171,86,205,25,0,134,246,239,223,175,106,106,106,178,1,49,1,236,64,252,67,
  73,32,216,0,44,6,238,70,174,174,158,246,50,13,133,66,115,14,194,208,208,144,106,105,105,201,85,57,62,11,124,157,18,106,133,6,0,219,15,52,
  2,109,136,179,113,152,117,118,118,206,57,0,74,137,95,216,183,111,159,170,171,171,203,6,194,20,240,115,45,127,209,214,96,80,48,55,195,203,144,154,
  219,247,128,87,108,70,21,21,21,106,120,120,120,94,64,80,74,169,104,52,170,158,122,234,41,85,94,94,158,13,136,151,108,199,88,40,8,54,0,11,
  144,10,241,189,192,3,64,39,146,106,58,76,182,109,219,54,95,250,59,116,225,194,5,245,224,131,15,102,3,161,25,57,33,22,12,130,13,64,166,109,
  240,87,155,65,85,85,149,26,29,29,45,88,216,75,151,46,169,29,59,118,168,149,43,87,170,71,30,121,68,29,56,112,64,165,211,233,146,65,72,167,
  211,106,215,174,93,153,172,97,63,80,107,131,80,16,0,234,230,109,208,128,92,75,109,215,123,204,97,210,222,222,94,144,144,111,188,241,134,170,174,174,
  190,105,149,58,58,58,84,50,153,44,25,4,165,148,218,179,103,143,119,254,24,240,69,36,79,112,124,66,49,0,216,209,96,53,240,99,224,77,175,18,
  221,221,221,57,5,235,237,237,205,86,215,47,10,196,124,244,234,171,175,102,10,143,171,145,139,158,37,133,88,129,23,0,115,85,190,92,35,249,3,36,
  222,198,108,70,149,149,149,234,212,169,83,25,133,10,135,195,170,182,182,54,87,232,82,128,122,225,133,23,74,6,224,201,39,159,244,206,251,46,208,10,
  124,9,57,71,84,144,39,60,122,99,162,177,130,37,26,197,38,224,81,96,47,114,224,112,152,53,52,52,168,137,137,137,25,2,197,227,241,108,183,190,
  163,200,105,211,233,171,169,169,41,233,254,241,242,229,203,153,182,216,107,192,143,128,111,32,31,117,45,202,183,13,50,1,80,166,247,79,45,240,101,36,
  53,126,2,56,238,85,172,185,185,89,69,163,81,71,249,245,235,215,103,82,254,10,114,185,217,239,5,177,165,165,197,183,83,124,248,225,135,189,124,146,
  192,111,129,14,36,26,220,163,183,115,206,109,144,41,51,10,32,167,48,99,5,15,0,63,1,126,129,28,72,102,48,110,106,106,82,253,253,253,106,245,
  234,213,153,148,31,7,14,3,127,214,171,51,232,125,102,223,190,125,69,43,191,123,247,238,76,188,78,3,191,1,182,32,39,200,123,145,176,158,19,128,
  108,223,9,6,45,16,238,210,147,213,35,135,165,71,117,127,62,186,161,173,102,82,255,78,105,97,190,143,156,59,0,168,171,171,227,252,249,243,84,87,
  87,23,48,37,28,60,120,144,182,182,54,111,165,42,10,28,66,106,25,23,144,143,61,34,72,21,105,10,72,43,175,162,154,110,170,168,234,7,211,136,
  73,77,33,158,245,26,114,30,143,0,7,52,195,92,52,5,252,29,41,90,140,235,247,71,117,51,81,5,144,43,184,103,158,121,38,207,116,66,199,143,
  31,103,243,230,205,94,229,141,243,139,35,137,91,92,243,79,82,64,161,36,87,73,57,141,56,174,235,8,146,87,181,2,151,145,52,121,50,203,123,147,
  200,202,127,128,128,119,21,169,226,68,144,45,244,30,114,177,225,208,115,207,61,199,197,139,23,115,10,122,240,224,65,214,174,93,75,60,30,247,14,157,
  65,252,76,84,243,139,34,32,76,107,29,178,174,62,100,1,64,191,160,16,179,189,161,39,29,67,14,72,255,66,110,145,94,66,170,71,6,101,133,172,
  244,59,8,96,166,140,53,140,128,246,190,254,123,5,185,216,152,54,252,166,167,167,217,186,117,43,217,228,220,179,103,15,109,109,109,153,148,191,0,92,
  68,194,244,56,174,197,153,82,89,202,251,130,151,110,242,1,206,128,123,207,102,162,194,82,36,182,222,131,248,132,187,17,255,240,73,36,131,52,229,245,
  27,136,21,124,128,128,53,172,127,39,112,83,237,251,144,111,145,191,99,243,220,181,107,23,219,183,111,119,254,29,14,135,233,236,236,164,167,167,39,147,
  136,17,196,244,39,144,133,9,35,78,54,130,44,68,76,131,144,211,2,178,126,45,174,148,82,26,4,179,21,98,136,197,216,95,109,25,95,97,194,77,
  74,63,247,161,165,252,8,82,202,190,161,1,80,250,239,235,72,152,93,110,120,238,220,185,147,242,242,114,234,235,235,233,237,237,229,249,231,159,207,84,
  38,7,89,245,115,200,246,252,80,43,124,149,153,181,194,100,62,229,33,135,5,56,15,8,8,38,52,86,33,150,80,11,124,10,73,54,106,117,223,66,
  205,116,82,11,52,130,248,140,49,100,11,153,27,221,59,16,235,169,7,190,137,196,237,178,156,66,184,164,128,127,34,219,41,138,0,123,5,89,253,48,
  110,181,56,129,172,190,202,7,64,206,255,47,0,142,37,160,21,48,119,135,38,241,48,81,98,9,2,64,202,18,204,92,94,152,48,104,172,101,129,30,
  95,130,124,90,255,54,240,237,252,186,115,29,113,160,87,113,173,108,20,177,52,187,82,124,67,243,201,171,60,20,0,0,56,32,24,5,18,150,50,9,
  4,0,83,136,48,125,81,45,140,9,73,246,39,120,113,253,206,24,2,194,155,184,22,145,137,210,200,138,95,210,115,27,179,31,69,162,202,176,6,197,
  118,126,5,223,19,228,221,2,51,30,118,147,164,50,100,197,43,116,43,199,189,101,154,214,109,10,215,19,27,129,76,130,85,141,108,29,147,96,221,7,
  124,30,248,42,226,100,211,90,33,19,126,99,122,62,179,242,87,113,125,140,49,123,115,77,150,4,39,146,229,165,130,44,192,144,101,9,222,100,201,222,
  195,41,171,165,245,107,182,67,53,239,76,34,86,176,24,177,160,114,221,183,20,247,218,91,33,32,38,44,229,175,225,154,253,85,4,164,24,238,29,97,
  193,202,23,13,128,153,92,251,4,163,96,146,153,31,39,216,249,185,35,140,7,132,143,112,19,44,99,65,101,122,204,68,139,50,205,195,40,111,50,74,
  147,144,141,233,190,235,88,23,164,197,40,15,62,0,176,149,210,64,100,100,152,73,16,143,67,53,86,80,134,171,252,20,18,37,170,116,159,89,253,9,
  100,245,199,116,51,137,150,241,246,190,175,199,124,1,96,43,228,231,29,203,161,198,113,173,199,228,26,53,8,0,65,235,153,40,178,218,227,8,104,215,
  17,75,113,178,61,63,178,64,137,0,148,72,230,74,30,102,38,91,139,144,109,17,208,207,76,33,32,152,102,156,171,115,216,241,171,60,20,25,5,102,
  147,60,169,246,2,220,168,178,16,55,219,52,183,214,211,136,210,182,226,190,76,254,38,57,230,11,0,152,1,130,73,173,141,63,176,63,224,76,227,126,
  196,97,34,80,73,171,62,67,134,249,4,192,17,194,5,34,96,53,112,29,236,77,81,101,214,120,223,14,0,24,202,245,165,231,108,43,238,240,188,157,
  0,152,15,42,233,35,195,255,7,250,216,3,240,95,74,157,253,8,108,254,206,192,0,0,0,0,73,69,78,68,174,66,96,130,0,0};
const char* MouseImages::mousecursor_png = (const char*)temp1;

static const unsigned char temp2[] = {137,80,78,71,13,10,26,10,0,0,0,13,73,72,68,82,0,0,0,64,0,0,0,64,8,6,0,0,0,170,105,113,222,0,0,0,4,115,66,73,
  84,8,8,8,8,124,8,100,136,0,0,0,9,112,72,89,115,0,0,13,215,0,0,13,215,1,66,40,155,120,0,0,0,25,116,69,88,116,83,111,
  102,116,119,97,114,101,0,119,119,119,46,105,110,107,115,99,97,112,101,46,111,114,103,155,238,60,26,0,0,10,139,73,68,65,84,120,156,213,155,127,
  104,149,215,25,199,63,239,125,111,147,92,141,75,213,68,59,145,198,154,24,136,113,172,50,212,173,163,58,24,173,236,159,98,100,136,162,147,49,166,34,
  182,129,66,116,10,226,44,235,10,54,148,253,49,24,66,43,164,131,178,105,22,107,149,174,108,12,38,166,254,4,197,97,42,26,92,150,217,216,89,123,
  213,107,98,111,140,185,247,158,253,113,222,115,239,115,223,31,55,247,87,18,251,192,33,239,57,231,125,239,57,207,247,60,231,249,117,78,44,165,20,19,
  65,150,101,89,238,54,53,81,131,149,64,86,185,230,228,199,112,46,122,82,192,40,11,0,130,249,70,224,151,64,11,240,52,240,31,224,83,224,143,192,
  24,160,68,1,158,0,32,148,82,37,21,192,2,66,192,111,129,81,178,153,52,229,223,192,143,128,74,224,41,32,236,124,99,225,44,194,84,149,114,49,
  191,55,128,113,89,238,2,75,128,233,14,16,79,4,8,229,96,254,59,64,194,48,250,226,139,47,170,195,135,15,171,83,167,78,169,142,142,14,85,91,
  91,43,65,248,39,48,7,168,1,34,142,52,76,41,8,165,48,111,57,171,248,43,195,224,11,47,188,160,18,137,132,146,116,225,194,5,101,89,150,1,
  32,225,72,193,60,180,142,136,72,73,152,10,0,66,197,107,15,66,78,249,158,105,216,184,113,35,182,109,103,189,180,108,217,50,22,45,90,100,170,54,
  176,12,248,22,48,13,168,112,218,74,153,71,73,84,240,192,66,227,27,9,152,109,250,22,46,92,232,251,205,115,207,61,39,171,207,162,87,223,232,130,
  167,156,223,178,10,53,165,229,160,98,145,55,251,191,152,213,171,1,102,2,213,64,21,217,202,112,210,169,20,0,44,138,3,96,58,122,11,72,107,96,
  59,191,87,176,67,85,42,21,52,121,151,248,135,208,147,47,116,194,17,52,243,17,180,14,144,150,96,210,169,20,229,99,64,40,116,226,21,104,209,175,
  36,163,4,45,81,38,149,202,97,5,10,157,116,152,140,55,104,152,255,230,88,1,135,228,86,40,20,0,169,60,37,136,223,56,9,40,118,178,134,209,
  41,219,247,146,74,21,189,82,65,40,229,55,202,66,83,181,247,220,214,68,110,131,73,53,133,225,201,26,72,144,133,246,6,23,3,35,192,53,224,31,
  192,0,26,8,5,164,36,8,202,9,64,38,130,2,1,8,88,5,183,216,22,179,82,47,163,125,0,73,163,192,239,129,223,56,207,73,68,40,61,145,
  233,53,15,0,98,176,217,192,15,208,43,53,6,156,7,46,162,35,186,98,77,32,120,153,7,237,19,180,3,115,129,29,104,0,34,232,204,210,151,232,
  204,146,1,5,4,40,165,2,145,6,64,48,110,66,220,61,232,136,77,210,127,129,173,192,25,50,166,172,96,16,170,171,171,105,109,109,165,174,174,142,
  147,39,79,114,233,210,37,211,245,51,52,216,141,192,247,209,142,18,64,12,248,3,240,59,224,1,206,54,161,28,64,184,226,251,16,208,69,238,204,78,
  18,216,9,60,3,44,68,131,161,0,245,201,39,159,40,63,90,189,122,117,250,251,185,115,231,170,129,129,129,116,95,42,149,82,27,54,108,24,47,155,
  100,202,101,160,142,140,27,93,114,86,73,90,1,11,189,186,63,53,13,45,45,45,180,181,181,177,121,243,102,102,205,154,101,154,67,192,155,232,76,80,
  21,5,90,146,182,182,54,234,235,235,51,131,90,22,59,118,236,240,125,183,161,161,129,218,218,90,217,244,93,224,128,51,174,204,37,132,156,223,42,42,
  164,150,129,77,63,14,218,219,182,109,83,201,100,50,189,82,183,110,221,82,75,151,46,149,171,113,17,88,1,92,160,0,9,56,113,226,132,167,63,26,
  141,102,173,244,238,221,187,85,127,127,191,82,74,169,68,34,161,222,121,231,29,217,255,8,104,64,167,214,102,146,73,172,100,73,3,121,74,132,1,32,
  12,204,55,131,68,34,17,21,143,199,61,19,237,237,237,85,21,21,21,114,50,191,6,46,21,2,128,223,59,247,239,223,87,114,108,119,90,77,41,165,
  234,235,235,229,184,127,5,62,71,235,139,40,208,13,252,88,0,97,231,11,132,68,172,201,136,68,115,115,51,145,136,87,89,183,180,180,240,218,107,175,
  201,166,159,83,102,95,162,162,162,194,147,86,3,104,106,106,146,213,159,160,23,204,100,164,214,2,127,71,155,81,9,66,122,107,4,141,103,0,8,161,
  205,13,0,253,253,253,129,19,220,181,107,23,211,166,165,141,195,2,4,112,19,73,161,208,184,170,38,4,236,2,126,129,87,71,4,130,96,0,176,129,
  59,192,48,64,44,22,227,218,181,107,190,163,204,153,51,199,173,180,42,243,228,161,108,20,137,68,248,232,163,143,136,199,227,92,185,114,133,151,94,122,
  73,118,191,9,124,27,175,165,240,5,65,2,16,70,43,52,0,222,123,239,189,192,9,236,220,185,147,233,211,167,151,206,73,145,180,125,251,118,94,121,
  229,21,34,145,8,75,150,44,225,248,241,227,210,178,212,0,171,201,228,28,159,34,135,36,72,71,198,6,142,153,142,206,206,78,70,71,71,125,39,80,
  87,87,23,104,186,38,131,150,47,95,158,85,175,170,170,98,205,154,53,178,105,41,217,153,103,147,186,243,149,0,68,199,57,244,86,224,238,221,187,116,
  119,119,7,78,162,189,189,125,202,164,160,166,166,198,211,214,220,220,44,171,207,162,77,228,12,178,165,32,16,0,67,73,180,54,5,224,224,193,131,129,
  147,168,171,171,227,213,87,95,205,127,214,19,76,143,31,63,150,85,139,12,0,230,8,206,55,243,108,194,79,227,226,38,129,191,161,253,108,122,122,122,
  184,122,245,106,224,160,237,237,237,84,87,87,167,235,225,240,84,68,215,154,134,134,134,220,77,53,120,245,128,103,27,24,0,12,243,99,192,23,104,159,
  27,128,3,7,14,4,14,90,91,91,203,219,111,191,141,109,219,44,88,176,128,149,43,87,150,204,72,177,52,60,60,44,171,10,237,33,70,200,62,137,
  246,216,210,144,248,32,9,60,70,187,154,199,205,11,31,124,240,1,215,175,95,15,28,120,251,246,237,12,12,12,112,227,198,13,42,42,42,124,223,89,
  181,106,21,160,149,85,67,67,131,167,95,58,62,121,216,123,95,114,1,144,34,227,11,200,251,8,16,32,1,41,116,156,255,24,157,165,185,136,206,212,
  144,76,38,217,191,127,127,206,193,231,207,159,239,235,189,25,122,253,245,215,233,238,238,230,236,217,179,242,160,52,77,51,102,204,160,181,181,21,128,117,
  235,214,229,28,43,136,92,91,32,129,102,90,122,132,129,86,192,13,192,35,224,107,224,136,121,233,200,145,35,244,246,246,22,53,49,208,43,191,118,237,
  90,158,127,254,249,192,119,142,30,61,202,192,192,64,160,226,109,108,108,76,63,207,156,57,211,211,239,146,128,4,94,198,125,61,193,48,217,74,112,12,
  13,64,28,248,23,240,25,208,146,74,165,216,178,101,11,61,61,61,19,170,232,100,152,236,166,61,123,246,16,10,133,88,188,120,49,43,86,172,240,244,
  187,0,24,43,100,92,19,11,84,160,205,198,60,116,172,255,50,250,234,75,250,114,211,222,189,123,125,163,189,39,129,154,154,154,100,180,248,103,96,55,
  176,1,248,33,58,124,158,141,214,11,54,34,66,148,74,48,133,150,130,81,180,30,120,8,92,7,78,24,164,222,122,235,45,122,122,122,10,1,119,82,
  40,26,141,210,215,215,103,170,10,184,135,147,50,115,21,15,133,148,74,231,210,220,219,224,33,48,4,124,12,220,0,72,165,82,108,218,180,137,88,44,
  54,49,156,20,73,103,206,156,145,213,123,104,93,102,248,113,3,145,69,210,230,24,9,72,160,1,24,65,71,135,15,208,247,252,70,0,110,222,188,201,
  214,173,91,203,205,67,73,228,2,224,14,154,23,119,145,11,157,166,16,100,101,84,165,63,16,71,75,193,3,116,246,229,47,230,163,174,174,46,14,29,
  58,84,102,54,138,167,211,167,79,203,234,109,50,11,105,36,64,74,65,22,185,189,14,183,20,152,109,112,31,29,40,93,52,47,182,181,181,229,116,144,
  38,139,6,7,7,57,127,254,188,108,146,0,184,65,240,80,26,128,0,41,144,219,224,30,90,10,238,1,196,227,113,214,175,95,31,24,50,79,22,117,
  116,116,48,54,150,182,122,81,180,15,51,38,74,130,28,122,192,207,239,116,75,193,215,104,0,98,232,180,217,159,156,126,46,95,190,204,251,239,191,95,
  86,134,10,161,59,119,238,240,238,187,239,202,166,171,100,28,186,199,100,64,144,202,48,139,178,0,16,82,32,61,195,56,90,10,238,163,87,255,51,32,
  189,233,186,186,186,202,194,76,49,212,209,209,193,200,200,136,169,198,128,65,244,156,71,209,139,55,74,102,27,164,153,23,124,122,111,139,139,88,217,156,
  21,84,162,51,43,79,163,115,241,207,160,111,123,238,5,29,2,15,15,15,83,85,85,85,94,238,198,161,193,193,65,26,27,27,229,22,60,135,246,91,
  190,4,110,1,55,209,128,124,133,150,224,17,156,237,32,1,240,108,1,151,20,24,191,192,56,70,49,180,36,220,192,73,160,38,18,9,110,223,190,93,
  102,246,198,167,55,222,120,67,50,31,67,51,252,136,140,43,63,66,102,43,200,211,230,44,242,141,61,3,182,130,81,136,49,167,60,48,239,71,163,209,
  82,249,41,136,250,250,250,232,236,236,148,77,87,201,14,228,12,0,143,200,182,4,30,0,2,35,27,165,148,114,118,131,219,69,30,70,39,27,210,241,
  231,100,123,134,251,246,237,35,145,72,152,234,87,232,36,142,97,254,161,83,226,232,57,27,37,168,192,123,138,156,79,104,39,183,130,17,175,97,180,201,
  1,224,208,161,67,159,126,248,225,135,33,160,210,182,109,59,28,14,219,182,109,91,225,112,216,10,135,195,33,219,182,17,207,166,221,50,207,182,109,103,
  61,251,181,201,250,177,99,199,106,201,132,183,189,98,94,15,241,2,32,205,160,135,198,253,151,25,71,41,154,179,3,163,16,103,1,203,129,45,104,199,
  227,115,116,18,210,28,70,200,24,92,17,16,139,143,67,238,111,100,125,5,80,139,190,86,115,14,45,141,81,180,36,12,2,255,35,91,249,141,1,73,
  247,234,67,158,255,51,228,128,96,14,79,166,161,19,142,115,208,161,243,60,244,205,14,115,82,43,211,79,249,50,152,139,89,119,155,201,237,217,100,246,
  122,140,204,86,248,2,109,9,238,161,37,193,72,129,242,3,160,144,236,134,113,39,199,200,40,69,227,42,87,162,87,58,142,55,255,238,150,128,92,204,
  6,221,61,178,92,197,0,156,32,3,192,61,231,239,16,121,138,63,20,6,128,204,25,72,171,80,229,76,234,17,90,252,101,170,221,143,114,93,180,202,
  213,39,143,241,204,115,210,153,199,16,25,0,30,34,196,222,153,115,224,21,154,98,242,91,70,10,12,0,33,167,62,36,126,207,29,126,6,73,64,174,
  43,183,238,203,148,102,229,109,50,82,166,208,139,241,16,237,159,24,0,30,57,115,202,185,250,144,39,0,142,73,180,240,58,71,33,167,109,132,140,242,
  51,171,111,162,47,63,37,88,140,20,184,1,48,219,64,58,106,102,245,141,243,147,115,245,161,112,9,48,204,25,231,200,76,192,156,197,195,56,25,24,
  252,153,14,234,115,3,19,114,21,179,40,198,9,26,33,207,189,159,254,209,124,172,64,250,229,140,73,148,43,145,245,31,31,100,175,124,46,0,220,207,
  227,189,231,182,2,238,67,29,163,156,147,162,140,123,125,174,24,29,32,197,219,108,9,183,214,247,123,150,84,168,95,224,214,5,110,63,195,88,168,172,
  228,199,120,204,67,129,18,0,158,11,6,238,67,7,223,188,91,190,63,93,224,59,126,160,167,197,62,31,230,161,8,0,210,31,78,193,191,184,185,167,
  32,158,179,152,200,151,121,128,255,3,130,90,183,183,86,183,3,85,0,0,0,0,73,69,78,68,174,66,96,130,0,0};
const char* MouseImages::mousecursorhand_png = (const char*)temp2;

static const unsigned char temp3[] = {137,80,78,71,13,10,26,10,0,0,0,13,73,72,68,82,0,0,0,64,0,0,0,64,8,6,0,0,0,170,105,113,222,0,0,0,4,115,66,73,
  84,8,8,8,8,124,8,100,136,0,0,0,9,112,72,89,115,0,0,13,215,0,0,13,215,1,66,40,155,120,0,0,0,25,116,69,88,116,83,111,
  102,116,119,97,114,101,0,119,119,119,46,105,110,107,115,99,97,112,101,46,111,114,103,155,238,60,26,0,0,10,9,73,68,65,84,120,156,237,155,123,
  104,20,219,29,199,63,59,179,217,100,53,214,170,217,216,94,228,70,72,12,196,216,86,165,104,75,65,255,40,86,74,65,84,138,168,88,75,11,42,65,
  13,8,209,26,16,171,125,88,52,72,255,43,130,6,162,69,218,106,227,243,182,210,254,37,228,170,215,90,197,98,174,104,144,52,181,209,70,239,154,172,
  73,238,110,146,125,76,255,56,115,118,206,206,99,179,187,121,172,112,253,193,33,51,231,156,157,57,191,239,239,125,230,196,103,24,6,95,100,210,138,189,
  128,98,211,123,0,138,189,128,98,211,123,0,138,189,128,98,211,123,0,138,189,128,98,211,123,0,138,189,128,98,211,23,30,0,255,100,63,208,231,243,
  249,178,141,27,239,88,238,237,155,172,245,184,48,238,6,68,250,101,239,10,16,19,6,192,198,184,15,203,172,74,129,184,50,102,40,45,149,238,44,50,
  16,19,242,1,10,243,62,132,57,253,24,184,13,12,0,67,64,55,240,123,96,49,2,144,18,115,158,110,254,102,92,147,153,114,50,12,163,160,102,50,
  32,37,62,7,248,152,76,41,171,45,14,28,0,230,1,95,2,130,64,0,11,8,95,161,235,152,104,43,216,4,76,201,249,76,38,254,4,108,80,199,
  75,74,74,136,199,85,11,32,9,252,8,184,9,140,154,45,110,246,23,205,36,10,50,1,69,109,53,224,187,40,204,55,55,55,243,234,213,43,98,177,
  24,247,238,221,99,197,138,21,114,72,7,126,129,208,150,114,160,12,161,5,14,147,152,86,179,40,80,237,165,228,75,129,223,96,170,250,214,173,91,13,
  59,245,245,245,25,129,64,64,117,126,63,0,182,2,63,55,1,217,14,124,136,229,31,52,219,123,166,212,60,114,101,214,222,52,115,193,51,128,143,36,
  0,231,206,157,115,0,96,24,134,177,124,249,114,213,31,132,113,250,136,24,112,28,161,25,170,86,56,192,152,108,80,28,137,144,162,126,243,128,111,35,
  60,120,28,184,11,220,7,18,138,6,148,152,139,6,32,20,10,185,106,153,173,127,158,203,148,50,132,147,252,10,176,7,225,23,130,64,61,240,10,248,
  183,217,39,253,131,33,215,57,81,159,145,6,64,97,220,15,252,12,104,70,72,88,165,255,0,59,128,91,8,0,2,20,224,71,202,203,203,217,176,97,
  3,161,80,136,155,55,111,242,224,193,3,57,180,29,1,112,13,240,45,243,249,0,17,224,119,192,111,129,183,88,230,52,113,32,108,234,174,1,23,241,
  14,103,6,66,18,77,64,8,88,128,18,254,110,220,184,225,106,2,107,215,174,77,255,126,254,252,249,70,79,79,79,122,44,149,74,25,91,182,108,201,
  246,62,181,61,52,223,43,195,104,134,207,40,196,4,84,233,249,128,157,192,15,101,71,125,125,61,141,141,141,108,223,190,157,185,115,231,202,110,13,248,
  53,240,117,96,38,121,106,64,99,99,35,85,85,85,214,75,125,62,118,239,222,237,58,183,186,186,154,138,138,10,181,235,27,8,95,33,35,136,244,19,
  154,249,172,130,34,136,148,188,31,145,185,25,128,177,107,215,46,35,153,76,166,37,245,226,197,11,99,217,178,101,170,52,254,9,124,19,248,7,121,104,
  192,245,235,215,29,227,225,112,56,67,210,7,15,30,52,186,187,187,13,195,48,140,68,34,97,156,60,121,82,29,31,1,170,129,74,68,72,157,129,139,
  54,144,163,70,72,0,252,8,117,54,0,35,24,12,26,209,104,212,177,208,206,206,78,53,164,25,192,97,132,99,204,25,0,183,57,3,3,3,134,250,
  238,68,34,225,152,83,85,85,165,190,247,175,192,127,17,206,57,12,180,35,242,17,123,94,49,46,16,42,98,181,82,37,234,234,234,8,6,131,14,85,
  169,175,175,103,239,222,189,106,215,79,16,145,96,210,40,16,8,160,235,186,163,191,182,182,86,189,253,62,66,96,126,68,84,217,8,252,29,248,37,206,
  48,154,181,222,144,0,104,136,112,3,64,119,119,183,231,2,15,28,56,192,140,25,233,224,176,16,5,184,169,36,77,27,215,213,104,136,80,250,83,156,
  62,194,19,4,9,128,14,188,70,84,112,68,34,17,158,60,121,226,250,150,202,202,74,187,211,42,205,145,135,73,163,96,48,200,213,171,87,137,70,163,
  60,122,244,136,53,107,214,168,195,191,2,190,138,51,82,184,130,160,2,224,71,56,52,0,206,156,57,227,185,128,253,251,247,51,115,230,204,137,115,82,
  32,53,52,52,176,110,221,58,130,193,32,75,150,44,225,218,181,107,106,100,153,13,172,197,170,55,74,200,162,9,18,0,204,73,87,228,64,91,91,27,
  163,163,163,174,11,8,133,66,158,161,107,58,72,41,176,0,40,43,43,99,253,250,245,106,215,50,224,203,136,48,93,138,16,174,244,117,25,36,13,75,
  14,124,130,48,5,222,188,121,67,123,123,187,231,34,154,154,154,138,166,5,179,103,207,118,244,213,213,213,169,183,31,34,66,228,44,50,181,192,19,0,
  73,73,132,55,5,224,212,169,83,158,139,8,133,66,236,217,179,39,247,85,79,49,141,141,141,169,183,62,44,0,130,216,0,80,205,64,35,51,197,77,
  2,127,195,220,160,232,232,232,224,241,227,199,158,47,109,106,106,162,188,60,93,11,225,247,79,250,38,115,206,52,56,56,104,239,154,141,211,15,56,204,
  64,2,32,153,143,3,47,17,57,55,0,199,143,31,247,124,105,69,69,5,39,78,156,64,215,117,22,46,92,200,170,85,171,38,204,72,161,52,52,52,
  164,222,26,136,12,49,136,229,3,210,41,179,74,154,242,131,36,48,134,72,53,175,201,9,231,207,159,231,233,211,167,158,47,110,104,104,160,167,167,135,
  103,207,158,17,8,4,92,231,172,94,189,26,16,206,170,186,186,218,49,174,38,62,57,196,123,87,178,1,144,194,202,5,212,141,22,240,208,128,20,162,
  12,29,67,108,78,220,7,158,0,36,147,73,142,28,57,146,245,229,11,22,44,112,205,222,36,237,219,183,143,246,246,118,238,220,185,195,162,69,139,28,
  227,179,102,205,98,195,6,177,171,182,105,211,166,172,239,242,34,155,9,36,16,76,187,109,172,100,144,27,0,35,192,231,192,5,57,233,194,133,11,116,
  118,118,22,180,48,16,146,223,184,113,35,75,151,46,245,156,115,233,210,37,122,122,122,60,29,111,77,77,77,250,122,206,156,57,142,113,155,6,36,112,
  223,81,114,144,159,76,39,24,71,0,16,5,254,5,124,10,212,167,82,41,118,236,216,65,71,71,199,148,58,58,181,76,182,83,115,115,51,154,166,177,
  120,241,98,86,174,92,233,24,183,1,16,119,76,200,66,178,22,8,32,194,198,7,192,215,128,239,1,135,204,135,25,128,113,232,208,33,215,106,239,93,
  160,218,218,90,181,90,252,35,112,16,216,2,124,7,81,62,207,67,248,5,29,165,66,84,157,96,10,161,5,163,8,63,48,12,60,5,174,75,164,142,
  29,59,70,71,71,71,62,224,78,11,133,195,97,186,186,186,228,173,1,244,99,110,153,217,154,131,52,195,72,239,165,217,205,96,24,24,4,254,2,60,
  3,72,165,82,108,219,182,141,72,36,50,53,156,20,72,183,111,223,86,111,251,17,190,76,242,99,7,34,131,212,152,35,53,32,129,0,32,134,168,14,
  223,2,231,204,123,158,63,127,206,206,157,59,39,155,135,9,145,13,128,215,8,94,236,77,21,116,154,52,200,216,81,85,243,129,40,66,11,222,34,118,
  95,254,44,127,116,241,226,69,90,91,91,39,153,141,194,233,214,173,91,234,109,31,150,32,165,6,168,90,144,65,246,172,195,174,5,210,12,6,16,133,
  210,125,57,177,177,177,49,107,130,52,93,212,219,219,203,221,187,119,213,46,21,0,59,8,14,74,3,224,161,5,170,25,244,35,180,160,31,32,26,141,
  178,121,243,102,207,146,121,186,168,165,165,69,253,8,27,70,228,48,113,165,37,200,226,7,220,242,78,187,22,124,142,0,32,130,216,54,251,131,57,206,
  195,135,15,57,123,246,236,164,50,148,15,189,126,253,154,211,167,79,171,93,143,177,18,186,49,44,16,84,103,152,65,25,0,40,90,160,102,134,81,132,
  22,12,32,164,255,41,226,203,16,32,252,65,177,168,165,165,133,88,44,38,111,35,64,47,98,205,163,8,225,141,98,153,129,235,241,28,199,249,0,219,
  167,111,63,162,154,154,137,216,97,169,68,124,191,91,130,72,146,240,251,253,12,13,13,81,86,86,54,185,220,141,67,189,189,189,212,212,212,168,38,248,
  9,34,111,121,5,188,0,158,35,0,249,12,161,193,49,76,115,80,1,112,152,128,77,11,100,94,32,19,163,8,66,19,158,97,110,160,38,18,9,250,
  250,250,38,153,189,241,233,232,209,163,42,243,17,4,195,35,88,169,124,12,203,20,164,6,140,27,5,0,79,83,144,14,49,98,182,183,114,126,56,28,
  158,40,63,121,81,87,87,23,109,109,109,106,215,99,50,11,57,9,192,8,153,145,192,1,128,103,101,99,24,134,97,90,131,61,69,30,66,108,54,164,
  235,207,233,206,12,15,31,62,76,34,145,144,183,159,33,54,113,36,243,195,102,139,146,121,12,199,0,231,87,228,92,74,59,213,20,164,122,13,33,66,
  14,0,173,173,173,31,95,190,124,89,3,74,117,93,215,253,126,191,174,235,186,207,239,247,251,252,126,191,166,235,58,202,181,236,247,201,107,93,215,51,
  174,221,250,212,251,43,87,174,84,96,149,183,157,202,186,134,113,2,160,134,65,7,141,123,72,202,118,24,74,58,196,185,192,10,196,89,129,62,68,166,
  56,7,235,99,132,90,131,27,120,212,226,227,80,182,131,151,43,129,10,160,7,225,252,6,17,2,121,137,112,124,255,35,211,249,197,129,164,93,250,144,
  227,65,73,19,4,249,241,100,6,98,195,177,18,81,58,127,0,204,199,250,82,171,110,63,229,202,96,46,167,76,213,232,164,153,235,145,182,30,193,50,
  133,151,136,72,208,143,208,4,169,5,134,27,0,249,236,110,200,116,50,142,229,20,101,170,92,138,144,116,20,231,254,187,93,3,178,49,235,203,50,199,
  126,70,9,4,99,18,128,126,243,239,32,57,170,63,228,7,128,186,103,160,70,133,50,115,81,35,88,71,102,60,235,111,156,76,102,3,64,29,83,63,
  227,201,235,164,185,142,65,44,0,134,81,212,222,92,179,231,17,154,66,246,183,164,22,72,0,52,243,126,80,121,158,189,252,244,210,0,149,81,47,169,
  171,215,82,245,165,150,25,8,97,12,35,242,19,9,192,136,185,166,172,210,135,28,1,48,67,162,15,103,114,164,153,125,49,44,231,39,165,47,171,47,
  55,39,88,136,22,216,1,144,102,160,38,106,82,250,50,249,201,42,125,200,95,3,36,115,50,57,146,11,144,223,226,97,156,29,24,220,153,246,26,179,
  3,163,217,154,20,138,76,130,98,228,104,251,233,135,230,18,5,210,147,173,144,168,74,34,227,168,43,153,146,207,6,128,253,122,188,121,246,40,96,255,
  168,35,157,115,82,105,227,30,159,43,196,7,168,234,45,77,194,238,245,221,174,85,202,55,47,176,251,2,123,158,33,35,84,198,230,199,120,204,67,158,
  26,0,142,3,6,246,143,14,174,251,110,185,62,58,207,57,110,160,167,213,62,23,230,161,0,0,210,63,44,246,63,58,120,107,93,94,167,70,255,15,
  217,69,234,231,137,127,244,14,0,0,0,0,73,69,78,68,174,66,96,130,0,0};
const char* MouseImages::mousecursorhandclosed_png = (const char*)temp3;
