//	HorizontalSlider.cpp - A horizontal slider widget.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#include "HorizontalSlider.h"
#include "drawing/Drawer.h"
#include "MouseCursor.h"

//------------------------------------------------------------------------------
HorizontalSlider::HorizontalSlider(const std::wstring& sliderImage):
Widget(),
ChangeBroadcaster(),
image(sliderImage),
value(0.0f),
handleSize(1.0f),
dragging(false),
mouseIsOver(false),
sizeIsSet(false)
{
	
}

//------------------------------------------------------------------------------
HorizontalSlider::~HorizontalSlider()
{
	
}

//------------------------------------------------------------------------------
void HorizontalSlider::draw(DrawerPtr& d)
{
	TwoFloats tempPos(getPosition());

	if(!sizeIsSet)
	{
		const Drawer::Image * const tempIm = d->getInternalImageData(image);
		if(tempIm)
		{
			handleSize = tempIm->size.x;
			sizeIsSet = true;
		}
	}

	tempPos.x += value * getSize().x;
	tempPos.x -= (handleSize*0.5f);

	d->drawImage(image, tempPos);
}

//------------------------------------------------------------------------------
void HorizontalSlider::mouseMove(const TwoFloats& pos)
{
	float left, right;
	const float w = getSize().x;

	mouseIsOver = true;
	MouseCursor::getInstance().setImage(true);

	left = getPosition().x;
	right = left + w;

	if(dragging)
	{
		if((pos.x >= left) && (pos.x < right))
			value = (pos.x-left)/w;
		else if(pos.x < left)
			value = 0.0f;
		else if(pos.x > right)
			value = 1.0f;
		callChangeCallbacks(this, value);
	}
}

//------------------------------------------------------------------------------
void HorizontalSlider::mouseButton(bool val, const TwoFloats& pos)
{
	float left, right;
	const float w = getSize().x;

	if(val)
	{
		left = getPosition().x;
		right = left + w;

		if((pos.x >= left) && (pos.x < right))
		{
			value = (pos.x-left)/w;
			dragging = true;
		}
	}
	else
	{
		dragging = false;
		callFinishedChanging(this);
	}
}

//------------------------------------------------------------------------------
void HorizontalSlider::mouseLeave(const TwoFloats& pos)
{
	float left, right;
	const float w = getSize().x;

	if(!dragging && mouseIsOver)
	{
		mouseIsOver = false;
		MouseCursor::getInstance().setImage(false);
	}
	else if(dragging)
	{
		left = getPosition().x;
		right = left + w;

		if((pos.x >= left) && (pos.x < right))
			value = (pos.x-left)/w;
		else if(pos.x < left)
			value = 0.0f;
		else if(pos.x > right)
			value = 1.0f;
		callChangeCallbacks(this, value);
	}
}

//------------------------------------------------------------------------------
void HorizontalSlider::setValue(float val)
{
	value = val;
}
