//	Callbacks.h - Base classes for various widget callbacks.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef CALLBACKS_H_
#define CALLBACKS_H_

#include <vector>
#include <string>

class Widget;

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
///	Called to trigger an action (e.g. after clicking a button).
class TriggerCallback
{
  public:
	///	Constructor.
	TriggerCallback() {};
	///	Destructor.
	virtual ~TriggerCallback() {};

	///	Called when something is triggered.
	/*!
		\param w The widget which triggered the callback.
		\param val The trigger value.
		\param id An additional identifier (if e.g. the widget has multiple
		triggers).
	 */
	virtual void triggered(Widget *w, bool val, const std::wstring& id) = 0;
};

//------------------------------------------------------------------------------
///	Called when a floating-point value changes (e.g. a slider).
class ChangeCallback
{
  public:
	///	Constructor.
	ChangeCallback() {};
	///	Destructor.
	virtual ~ChangeCallback() {};

	///	Called when a floating-point value is changed.
	/*!
		\param w The widget which triggered the callback.
		\param val The new value.
		\param id An additional identifier (if e.g. the widget has multiple
		values which may change).
	 */
	virtual void valueChanged(Widget *w, float val, const std::wstring& id) = 0;
	///	Called when the user has finished interacting with the widget for now.
	/*!
		\param w The widget which triggered the callback.
	 */
	virtual void finishedChanging(Widget *w) {};
};

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
///	Base class for objects which call TriggerCallbacks.
class TriggerBroadcaster
{
  public:
	///	Constructor.
	TriggerBroadcaster();
	///	Destructor.
	virtual ~TriggerBroadcaster();

	///	Sends a trigger message to any registered callbacks.
	void callTriggerCallbacks(Widget *w, bool val, const std::wstring& id=L"");

	///	Adds a callback.
	void addTriggerCallback(TriggerCallback *c);
	///	Removes a callback.
	void removeTriggerCallback(TriggerCallback *c);
  private:
	///	All the registered callbacks.
	std::vector<TriggerCallback *> callbacks;
};

//------------------------------------------------------------------------------
///	Base class for objects which call ChangeCallbacks.
class ChangeBroadcaster
{
  public:
	///	Constructor.
	ChangeBroadcaster();
	///	Destructor.
	virtual ~ChangeBroadcaster();

	///	Sends a change message to any registered callbacks.
	void callChangeCallbacks(Widget *w, float val, const std::wstring& id=L"");
	///	Sends a 'finishedChanging' message to any registered callbacks.
	void callFinishedChanging(Widget *w);

	///	Adds a callback.
	void addChangeCallback(ChangeCallback *c);
	///	Removes a callback.
	void removeChangeCallback(ChangeCallback *c);
  private:
	///	All the registered callbacks.
	std::vector<ChangeCallback *> callbacks;
};

#endif
