//	FontAtlas.h - Base class for the different font renderers.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef FONTATLAS_H_
#define FONTATLAS_H_

#include "FilePath.h"
#include "c0xHeaders.h"
#include "FontMetrics.h"
#include "TextureAtlas.h"

class FontAtlas;
typedef shared_ptr<FontAtlas> FontAtlasPtr;

///	Base class for the different font renderers.
/*!
	In case we stop using freetype for everything.
 */
class FontAtlas : public TextureAtlas
{
  public:
	///	Constructor.
	FontAtlas();
	///Destructor.
	virtual ~FontAtlas();

	///	Used to load a glyph missing from the passed-in FontMetrics.
	/*!
		\param missingCharacter The glyph to load.
		\param metrics The metrics to add the new glyph data to.
		\param atlasImage On return this will point to the new AtlasImage, so
		you can load it into your images array correctly. Note that you will
		want to fill out the atlasName member of the returned atlasImage
		according to the correct atlas. *DO NOT DELETE THIS* it is owned by
		the FontAtlas.
		\param imageData On return this will contain the image data to load
		into the existing texture atlas. The memory will be allocated for you.
		It is assumed this will be deleted by the caller.
		\param imageWidth On return this will contain the width of the image
		data.
		\param imageHeight On return this will contain the height of the image
		data.
		\param needNewAtlas On return this will be true if a new atlas is
		needed to accommodate the new glyph. Note: if this is true, the returned
		imageData will be the size of the atlas, not the size of the glyph.

		If the relevant glyph could not be loaded, *atlasImage, *imageData,
		imageWidth, and imageHeight will all be 0.
	 */
	virtual void getMissingGlyphData(wchar_t missingCharacter,
									 FontMetrics& metrics,
									 AtlasImage **atlasImage,
									 unsigned char **imageData,
									 int& imageWidth,
									 int& imageHeight,
									 bool& needNewAtlas) = 0;

	///	Returns the dimensions of the atlas texture.
	/*!
		FontAtlases are assumed to be square.
	 */
	virtual int getAtlasDimensions() const = 0;

	///	Used to find the location of a font file.
	/*!
		First checks the system font directory, then if the font wasn't there,
		checks DataFilesDir/fonts/

		\sa FilePath::getDataFilesDir()
	 */
	static FilePath findFontFile(const std::wstring& name);

	///	Used to construct a FontAtlas subclass.
	/*!
		All FontAtlas subclasses should implement one of these.

		\sa Drawer::setFontRenderer()
	 */
	class Creator
	{
	  public:
		///	Destructor.
		virtual ~Creator() {};

		///	Returns a new instance of a FontAtlas subclass.
		virtual FontAtlasPtr createInstance(const FilePath& fontFile,
											float fontSize,
											FontMetrics& metrics,
											const std::wstring& instanceName) = 0;
	};
};

#endif
