//	TextureAtlas.h - Base class etc. for texture atlases.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef TEXTUREATLAS_H_
#define TEXTUREATLAS_H_

#include "TwoFloats.h"

#include <string>

//------------------------------------------------------------------------------
///	Struct representing an image within the atlas.
struct AtlasImage
{
	///	Name of the image, derived from its filename minus the extension.
	std::wstring name;
	///	Name of the texture atlas, derived from its filename minus the extension.
	std::wstring atlasName;
	///	Position of the image within the atlas (0->1).
	TwoFloats position;
	///	Size of the image relative to the atlas (0->1).
	TwoFloats size;

	///	Constructor.
	AtlasImage(const std::wstring& imageName,
			   const std::wstring& textureAtlasName,
			   const TwoFloats& imagePos,
			   const TwoFloats& imageSize):
	name(imageName),
	atlasName(textureAtlasName),
	position(imagePos),
	size(imageSize)
	{
		
	};
	///	Default constructor.
	AtlasImage()
	{

	};
};

//------------------------------------------------------------------------------
///	Base class for all texture atlases.
/*!
	Subclasses will be created by the ImageResourceBuilder app; you won't want
	to subclass this by hand.
 */
class TextureAtlas
{
  public:
	///	Constructor.
	TextureAtlas(char *data,
				 int dataSize,
				 int sizeImageArray,
				 AtlasImage *imageArray):
	imageData(data),
	imageDataSize(dataSize),
	numImages(sizeImageArray),
	images(imageArray)
	{
		
	};
	///	Destructor.
	virtual ~TextureAtlas() {};

	///	Returns the image data for this atlas.
	const char *getImageData() const {return const_cast<const char *>(imageData);};
	///	Returns the size of the image data for this atlas.
	const int getImageDataSize() const {return imageDataSize;};

	///	Returns the number of images within this atlas.
	const int getNumImages() const {return numImages;};
	///	Returns the indexed AtlasImage.
	const AtlasImage& getImage(int index) const {return const_cast<const AtlasImage&>(images[index]);};
  protected:
	///	Used by FontAtlas subclasses.
	void initialise(const char *data,
					int dataSize,
					int sizeImageArray,
					AtlasImage *imageArray)
	{
		imageData = const_cast<char *>(data);
		imageDataSize = dataSize;
		numImages = sizeImageArray;
		images = imageArray;
	};
  private:
	///	The image data for this atlas.
	char *imageData;
	///	The size of the image data for this atlas.
	int imageDataSize;

	///	The number of images within this atlas.
	int numImages;
	///	The images within this atlas.
	AtlasImage *images;
};

#endif
