//	SDL2.0AudioDevice.h - An SDL 2.0 subclass of AudioDevice.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef SDL2_0AUDIODEVICE_H_
#define SDL2_0AUDIODEVICE_H_

#include "sound/AudioDevice.h"

#include "SDL_audio.h"
#include <stdint.h>

///	An SDL 2.0 subclass of AudioDevice.
class SDL2_0AudioDevice : public AudioDevice
{
  public:
	///	Constructor.
	SDL2_0AudioDevice();
	///	Destructor.
	~SDL2_0AudioDevice();

	///	Sets the callback to call during audio processing.
	void setCallback(AudioDeviceCallback *val);

	///	Sets the number of desired inputs and outputs.
	/*!
		Note: The actual number of channels *may* be different to what you
		pass in here, depending on what SDL_OpenAudio() returns. You may
		want to call getNumInputs() and getNumOutputs() after you call
		start().
	 */
	void setNumChannels(int inputs, int outputs);
	///	Sets the desired samplerate.
	void setSamplerate(double rate);
	///	Not used.
	void setAPI(const std::wstring& val) {};
	///	Sets the audio devices to use for input and output.
	void setDevices(const std::wstring& inDevice,
					const std::wstring& outDevice) {};

	///	Starts audio processing.
	/*!
		\return True if successful, false if there's a problem.
	 */
	bool start();
	///	Stops audio processing.
	void stop();

	///	Returns the number of input channels.
	int getNumInputs() const {return numInputs;};
	///	Returns the number of output channels.
	int getNumOutputs() const {return numOutputs;};
	///	Returns the samplerate.
	double getSamplerate() const {return samplerate;};
	///	Returns the current API.
	const std::wstring getAPI() const {return L"SDL 2.0";};

	///	Returns the available audio APIs as a vector of strings.
	const std::vector<std::wstring>& getAvailableAPIs() const {return apiList;};
	///	Returns the available inputs for the selected API (selected via setAPI()).
	const std::vector<std::wstring>& getAvailableInputDevices() const {return inputList;};
	///	Returns the available outputs for the selected API (selected via setAPI()).
	const std::vector<std::wstring>& getAvailableOutputDevices() const {return outputList;};
	///	Returns the number of available channels for the input device (index from getAvailableInputDevices()).
	int getChannelsForInputDevice(int index) const {return 2;};
	///	Returns the number of available channels for the output device (index from getAvailableOutputDevices()).
	int getChannelsForOutputDevice(int index) const {return 2;};

	///	Our SDL audio callback.
	static void sdlAudioCallback(void *user, uint8_t *buffer, int numBytes);

	///	Called from sdlAudioCallback.
	AudioDeviceCallback *getCallback() {return callback;};
  private:
	///	The audio spec structure describing our current settings.
	SDL_AudioSpec *audioSpec;
	///	A temporary buffer used to hold the converted integer audio data.
	/*
		SDL 1.2 is so old-fashioned! No float sample format?!

		Is making this static going to cause problems?
	 */
	static float *floatBuffer;
	///	Our callback.
	AudioDeviceCallback *callback;

	///	The number of input channels.
	int numInputs;
	///	The number of output channels.
	int numOutputs;
	///	The samplerate.
	double samplerate;

	///	Returned by getAvailableAPIs().
	std::vector<std::wstring> apiList;
	///	Returned by getAvailableInputs().
	std::vector<std::wstring> inputList;
	///	Returned by getAvailableOutputs().
	std::vector<std::wstring> outputList;

	///	Whether we're currently running or not.
	bool isRunning;
};

#endif
