//	ResourceManager.h - Class used to manage the various kinds of resource we
//						might use.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef RESOURCEMANAGER_H_
#define RESOURCEMANAGER_H_

#include "sound/SoundFileLoader.h"
#include "c0xHeaders.h"
#include "FilePath.h"

#include <vector>

class TextureAtlas;
class SingleImage;
class Shader;
class Font;

typedef shared_ptr<TextureAtlas> TextureAtlasPtr;
typedef shared_ptr<SingleImage> SingleImagePtr;
typedef shared_ptr<Shader> ShaderPtr;
typedef shared_ptr<Font> FontPtr;

///	Class used to manage the various kinds of image we might use.
class ResourceManager
{
  public:
	///	Constructor.
	ResourceManager();
	///	Destructor.
	~ResourceManager();

	///	Method implemented by the AVLibResourceGenerator.
	/*!
		This method is used by the AVLibResourceGenerator to inform the
		ResourceManager of the various images etc. it needs to take care of.

		Essentially this breaks down to a series of registerTextureAtlas(),
		registerSingleImage(), registerImageFolder() calls.
	 */
	void initialiseResources();

	///	Registers a TextureAtlas object with the ResourceManager.
	void registerTextureAtlas(TextureAtlasPtr atlas);
	///	Registers a SingleImage object with the ResourceManager.
	void registerSingleImage(SingleImagePtr image);
	///	Registers an image folder with the ResourceManager.
	void registerImageFolder(const FilePath& dir);
	///	Registers a shader with the ResourceManager.
	void registerShader(ShaderPtr shader);
	///	Registers a font with the ResourceManager.
	void registerFont(FontPtr font);
	///	Registers a sound file with the ResourceManager.
	void registerSound(const FilePath& sound);
	///	Registers a sound folder with the ResourceManager.
	void registerSoundFolder(const FilePath& dir);

	///	Registers a SoundFileLoader type.
	/*
		\param extension The file extension of sound files this loader will
		load.
		\param creator Pointer to a SoundFileLoader::Creator instance which
		will instantiate the relevant SoundFileLoader. It is the caller's
		responsibility to delete this when they are finished with it (typically,
		you'd create it as a static member of your SoundFileLoader subclass so
		you don't have to new and delete it manually).

		The VorbisfileLoader is registered by default. To register other
		loaders, call Application::getResourceManager().registerSoundFileLoader()
		in your app's AppSettings::initialise() method.
	 */
	void registerSoundFileLoader(const std::wstring& extension,
								 SoundFileLoader::Creator *creator);

	///	Returns the number of registered TextureAtlases.
	int getNumTextureAtlases() const {return atlases.size();};
	///	Returns the indexed TextureAtlas.
	TextureAtlasPtr getTextureAtlas(int index) const {return atlases[index];};

	///	Returns the number of registered SingleImages.
	int getNumSingleImages() const {return images.size();};
	///	Returns the indexed TextureAtlas.
	SingleImagePtr getSingleImage(int index) const {return images[index];};

	///	Returns the number of registered image folders.
	int getNumFolders() const {return folders.size();};
	///	Returns the indexed image folder.
	const FilePath& getFolder(int index) const {return folders[index];};

	///	Returns the number of registered Shaders.
	int getNumShaders() const {return shaders.size();};
	///	Returns the indexed TextureAtlas.
	ShaderPtr getShaders(int index) const {return shaders[index];};

	///	Returns the number of registered fonts.
	int getNumFonts() const {return fonts.size();};
	///	Returns the indexed font.
	FontPtr getFont(int index) const {return fonts[index];};

	///	Returns the number of registered sounds.
	int getNumSounds() const {return sounds.size();};
	///	Returns a SoundFileLoader for the indexed sound.
	/*!
		Note that the returned SoundFileLoaderPtr may be empty if no loader
		could be found to match the indexed file's extension.
	 */
	SoundFileLoaderPtr getSound(int index) const {return soundFileFactory.getLoader(sounds[index]);};
	///	Returns a SoundFileLoader for the sound at the passed-in path.
	/*!
		Note that the returned SoundFileLoaderPtr may be empty if no loader
		could be found to match the indexed file's extension.
	 */
	SoundFileLoaderPtr getSound(const FilePath& path) const  {return soundFileFactory.getLoader(path);};

	///	Returns the number of registered sound folders.
	int getNumSoundFolders() const {return soundFolders.size();};
	///	Returns the indexed sound folder.
	const FilePath& getSoundFolder(int index) const {return soundFolders[index];};
  private:
	///	All the registered TextureAtlases.
	std::vector<TextureAtlasPtr> atlases;
	///	All the registered SingleImages.
	std::vector<SingleImagePtr> images;
	///	All the registered image folders.
	std::vector<FilePath> folders;
	///	All the registered Shaders.
	std::vector<ShaderPtr> shaders;
	///	All the registered fonts.
	std::vector<FontPtr> fonts;
	///	All the registered sounds.
	std::vector<FilePath> sounds;
	///	All the registered sound folders.
	std::vector<FilePath> soundFolders;

	///	Used to determine which SoundFileLoader matches which sound file extension.
	SoundFileFactory soundFileFactory;
};

#endif
