//	LoadingScreen.h - Loading screen state.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef LOADINGSCREEN_H_
#define LOADINGSCREEN_H_

#include "ThreeFloats.h"
#include "GameState.h"

///	Loading screen state.
/*!
	This state loads all the game's resources and displays its current progress
	in doing so.
 */
class LoadingScreen : public GameState
{
  public:
	///	Constructor.
	LoadingScreen(Application& app, SounderPtr sound);
	///	Destructor.
	~LoadingScreen();

	///	Draws the current state.
	void draw(DrawerPtr& d, float timestepInterp);

	///	Used to quit the game with escape.
	void keyDown(const std::wstring& action,
				 const std::wstring& keyPressed,
				 int keyModifier);

	///	Sets the colours to use.
	void setColours(const ThreeFloats * const col);

	///	Returns the name of this state.
	const std::wstring getName() const {return L"LoadingScreen";};
	///	Used to create instances of this state.
	static GameStateCreator<LoadingScreen> creator;
  private:
	///	Enum naming each of the resource types we're loading.
	enum ResourceType
	{
		TextureAtlasRes = 0,
		SingleImageRes,
		ImageFolderRes,
		ShaderRes,
		FontRes,
		SoundRes,
		SoundFolderRes,

		NumResourceTypes
	};

	///	IDs of the various colours we use.
	enum
	{
		Background = 0,
		PixelBright,
		PixelDark,
		EndFade,

		NumColours
	};
	///	The size of the 'pixel' grid we use.
	enum
	{
		GridWidth = 16,
		GridHeight = 9
	};

	///	The total number of resources we have to load.
	int totalResources;
	///	Our current progress in loading the resources.
	int resProgress;
	///	Which resource type we're currently loading.
	ResourceType resType;
	///	Index of the resource type we're currently loading.
	int resIndex;
	///	Subindex of the resource type we're currently loading (for folder resource types)
	int subIndex;

	///	Used to load any sound files.
	SounderPtr sounder;

	///	The various colours we use.
	ThreeFloats colours[NumColours];

	///	Our grid of 'pixels'.
	float grid[GridHeight][GridWidth];
	///	Used to draw the pixel spiral.
	float spiralLastVal;

	///	Used to fade the screen out once it's done.
	float fadeOut;
};

#endif
