//	KeyManager.h - Class handling the mapping of keys to actions.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef KEYMANAGER_H_
#define KEYMANAGER_H_

#include <string>
#include <vector>
#include <map>

///	Class handling the mapping of keys to actions.
/*!
	Generally you'll want to subclass this to add methods for mapping
	backend-specific events to actions.
 */
class KeyManager
{
  public:
	///	Constructor.
	KeyManager();
	///	Destructor.
	virtual ~KeyManager();

	///	Sets up a mapping between an action and a key/event.
	/*!
		\param action The action to be mapped to.
		\param key The key to map to the above action.
		\param isDefault If this is true, the mapping will only be added if
		there are no existing mappings for this action.
	 */
	void addMapping(const std::wstring& action,
					const std::wstring& key,
					bool isDefault = false);

	///	Returns the action for the passed-in key/event.
	/*!
		\param key Essentially this can be anything, though you'll probably want
		to name it after the key, joystick button etc.
		\param actions A vector which, upon return, will be full of all the
		actions mapped to this key/event.
	 */
	void getActions(const std::wstring& key,
					std::vector<std::wstring>& actions) const;
  private:
	///	Helper method. Loads an initial set of mappings from keyconfig.txt (if it exists).
	void loadMappings();
	///	Helper method. Saves the current mappings to keyconfig.txt.
	void saveMappings();

	///	Our mappings <key, action>.
	std::multimap<std::wstring, std::wstring> mappings;

	///	True if there was no keyconfig file when KeyManager was constructed.
	bool createFile;
};

#endif
