//	KeyManager.cpp - Class handling the mapping of keys to actions.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#include "AppSettings.h"
#include "KeyManager.h"
#include "UTF8File.h"

#include <sstream>

using std::wstringstream;
using std::multimap;
using std::wstring;
using std::vector;
using std::endl;

//------------------------------------------------------------------------------
KeyManager::KeyManager():
createFile(false)
{
	loadMappings();
}

//------------------------------------------------------------------------------
KeyManager::~KeyManager()
{
	if(createFile)
		saveMappings();
}

//------------------------------------------------------------------------------
void KeyManager::addMapping(const wstring& action,
							const wstring& key,
							bool isDefault)
{
	if(isDefault)
	{
		bool dontAdd = false;
		multimap<wstring, wstring>::iterator it;

		if(!createFile)
		{
			for(it=mappings.begin();it!=mappings.end();++it)
			{
				if(it->second == action)
				{
					dontAdd = true;
					break;
				}
			}
		}
		if(!dontAdd)
			mappings.insert(make_pair(key, action));
	}
	else
		mappings.insert(make_pair(key, action));
}

//------------------------------------------------------------------------------
void KeyManager::getActions(const wstring& key, vector<wstring>& actions) const
{
	multimap<wstring, wstring>::const_iterator it;

	actions.clear();

	for(it=mappings.lower_bound(key);
		it!=mappings.upper_bound(key);
		++it)
	{
		actions.push_back(it->second);
	}
}

//------------------------------------------------------------------------------
void KeyManager::loadMappings()
{
	FilePath filePath = FilePath::getAppDataPath().getChild(AppSettings::applicationName).getChild(L"keyconfig.txt");
	
	if(filePath.exists())
	{
		UTF8File mappingsFile(filePath);
		wstring tempstr;
		wstring action;
		wstring key;

		while(!mappingsFile.readLine(tempstr))
		{
			action = tempstr.substr(0, tempstr.find_first_of(L' '));
			key = tempstr.substr(tempstr.find_first_of(L' ')+1);

			mappings.insert(make_pair(key, action));
		}
	}
	else
		createFile = true;
}

//------------------------------------------------------------------------------
void KeyManager::saveMappings()
{
	wstringstream tempstr;
	multimap<wstring, wstring>::iterator it;
	FilePath filePath = FilePath::getAppDataPath().getChild(AppSettings::applicationName).getChild(L"keyconfig.txt");
	UTF8File mappingsFile(filePath);

	for(it=mappings.begin();it!=mappings.end();++it)
	{
		tempstr.str(L"");
		tempstr.clear();

		tempstr << it->second << L" " << it->first << endl;

		mappingsFile.writeText(tempstr.str());
	}
}

