//	ConfigFile.h - Class used to store and load application config details.
//  ----------------------------------------------------------------------------
//	This file is part of 'NiallsAVLib', base code for any kind of audiovisual
//	apps.
//	Copyright (C) 2012  Niall Moody
//	
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.
//	----------------------------------------------------------------------------

#ifndef CONFIGFILE_H_
#define CONFIGFILE_H_

#include "CmdLineParams.h"
#include "UTF8File.h"

#include <map>

///	Class used to store and load application config details.
class ConfigFile
{
  public:
	///	Constructor.
	/*!
		Sets up the available config parameters, and if it exists, loads the
		previous config file from disk.
	 */
	ConfigFile(const CmdLineParams& cmdLine);
	///	Destructor.
	/*!
		Saves the current config parameters to the config file on disk.
	 */
	~ConfigFile();

	///	Sets the value of the named parameter.
	/*!
		\param token This is the short token of the parameter the caller is
		interested in (config parameters are stored as a shortToken-value
		pair in the config file).
		\param value This is the value of the named parameter. Note that for
		hasValue=false parameters, this should be either "1" or "0" to indicate
		the presence or otherwise of the parameter.
	 */
	void setParameterValue(const std::wstring& token, const std::wstring& value);

	///	Returns the value of the named parameter.
	/*!
		\param token This is the short token of the parameter the caller is
		interested in (config parameters are stored as a shortToken-value
		pair in the config file).

		\return Will return an empty string if no value was found in the config
		file.
	 */
	const std::wstring& getParameterValue(const std::wstring& token) {return parameters[token];};

	///	Returns true if the config file existed when this instance was created.
	bool exists() const {return existing;};
  private:
	///	The actual file which is written to/read from.
	UTF8File phil;

	///	The available parameters for this app.
	std::map<std::wstring, std::wstring> parameters;

	///	Whether or not the config file existed when this instance was created.
	bool existing;
};

#endif
